#version 140
#extension GL_EXT_gpu_shader4 : enable

uniform sampler2D texture0;        // dancer feed
uniform vec2 textureSize0;         // resolution of dancer texture
uniform float depthAmount;         // max pixel shift in pixels (default 5.0)
uniform float maskSoftness;        // edge mask softness (default 0.15)
uniform float gradientStrength;    // cylindrical gradient strength (default 0.8)
uniform float grayBlend;           // blend between cyl (0.0) and grayscale (1.0), default 0.5
uniform float eyeSign;             // +1.0 = right eye, -1.0 = left eye

varying vec4 gl_TexCoord[];

void main(void) {
    vec2 uv = gl_TexCoord[0].xy;

    // --- Defaults if not passed ---
    float depthAmt     = (depthAmount      != 0.0) ? depthAmount      : 5.0;
    float maskSoft     = (maskSoftness     != 0.0) ? maskSoftness     : 0.15;
    float gradStrength = (gradientStrength != 0.0) ? gradientStrength : 0.8;
    float grayMix      = (grayBlend        != 0.0) ? grayBlend        : 0.5;
    float eyeDir       = (eyeSign          != 0.0) ? eyeSign          : 1.0;

    // --- Sample neighbors (with .rgba swizzle) ---
    vec4 rgbNW = texture2D(texture0, uv + vec2(-1.0, -1.0) / textureSize0).rgba;
    vec4 rgbNE = texture2D(texture0, uv + vec2( 1.0, -1.0) / textureSize0).rgba;
    vec4 rgbSW = texture2D(texture0, uv + vec2(-1.0,  1.0) / textureSize0).rgba;
    vec4 rgbSE = texture2D(texture0, uv + vec2( 1.0,  1.0) / textureSize0).rgba;
    vec4 rgbM  = texture2D(texture0, uv).rgba;

    float alphaCenter = rgbM.a;

    // --- Alpha-based edge detection for outer contour only ---
    float alphaMin = min(alphaCenter, min(min(rgbNW.a, rgbNE.a), min(rgbSW.a, rgbSE.a)));
    float alphaMax = max(alphaCenter, max(max(rgbNW.a, rgbNE.a), max(rgbSW.a, rgbSE.a)));
    float edgeStrength = alphaMax - alphaMin;

    // Mask: 1.0 inside, fade to 0.0 at outer contour
    float insideMask = 1.0 - smoothstep(0.0, maskSoft, edgeStrength);

    // --- Cylindrical gradient ---
    float cyl = clamp(1.0 - abs(uv.x - 0.5) * 2.0, 0.0, 1.0);
    cyl = mix(1.0, cyl, gradStrength);

    // --- Grayscale of the model ---
    float gray = dot(rgbM.rgb, vec3(0.299, 0.587, 0.114));

    // --- Blend grayscale with cylindrical gradient for fake depth ---
    float blendedDepth = mix(cyl, gray, grayMix);

    // --- Final depth mask ---
    float depthMask = insideMask * blendedDepth;

    // --- Pixel shift based on eyeSign ---
    float maxShift = depthAmt / textureSize0.x;
    vec2 offset = vec2(eyeDir * (depthMask - 0.5) * maxShift, 0.0);

    vec4 eyeView = texture2D(texture0, uv + offset).rgba;

    // --- Apply transparency outside edges ---
    float outAlpha = alphaCenter * insideMask;
    eyeView.rgb *= insideMask;
    eyeView.a = outAlpha;

    gl_FragColor = eyeView.rgba;
}
