#version 140
#extension GL_EXT_gpu_shader4 : enable
// version 08
uniform sampler2D texture0;        // dancer feed
uniform vec2 textureSize0;         // resolution of dancer texture

uniform float depthAmount;         // max pixel shift in pixels (default 5.0)
uniform float maskSoftness;        // alpha edge softness (default 0.15)
uniform float gradientStrength;    // cylindrical gradient strength (default 0.8)
uniform float grayBlend;           // blend between cyl (0.0) and grayscale (1.0), default 0.5
uniform float eyeSign;             // +1.0 = right eye, -1.0 = left eye
uniform float debugMode;           // 0.0 = normal, 1.0 = blended mask, 2.0 = cyl-only mask

varying vec4 gl_TexCoord[];

void main(void) {
    vec2 uv = gl_TexCoord[0].xy;

    // --- Defaults if not passed ---
    float depthAmt     = (depthAmount      != 0.0) ? depthAmount      : 5.0;
    float maskSoft     = (maskSoftness     != 0.0) ? maskSoftness     : 0.15;
    float gradStrength = (gradientStrength != 0.0) ? gradientStrength : 0.8;
    float grayMix      = (grayBlend        != 0.0) ? grayBlend        : 0.5;
    float eyeDir       = (eyeSign          != 0.0) ? eyeSign          : 1.0;

    // --- Sample center pixel ---
    vec4 rgbM  = texture2D(texture0, uv).rgba;
    float alphaCenter = rgbM.a;

    // --- Flood-fill mask from alpha ---
    float insideMask = smoothstep(0.0, maskSoft, alphaCenter);

    // --- Cylindrical gradient ---
    float cyl = clamp(1.0 - abs(uv.x - 0.5) * 2.0, 0.0, 1.0);
    cyl = mix(1.0, cyl, gradStrength);

    // --- Grayscale luminance of the model ---
    float gray = dot(rgbM.rgb, vec3(0.299, 0.587, 0.114));

    // --- Blend grayscale with cylindrical gradient ---
    float blendedDepth = mix(cyl, gray, grayMix);

    // --- Final depth mask ---
    float depthMask = insideMask * blendedDepth;
    float cylMask   = insideMask * cyl;

    // --- Debug modes (ignore stereo offset) ---
    if (debugMode > 1.5) {
        // Mode 2: pure cylindrical mask — no double alpha dimming
        gl_FragColor = vec4(vec3(cyl), 1.0); // full alpha for visibility
        return;
    }
    if (debugMode > 0.5) {
        // Mode 1: blended mask — no double alpha dimming
        gl_FragColor = vec4(vec3(blendedDepth), 1.0);
        return;
    }

    // --- Normal stereo mode ---
    float maxShift = depthAmt / textureSize0.x;
    vec2 offset = vec2(eyeDir * (depthMask - 0.5) * maxShift, 0.0);

    vec4 eyeView = texture2D(texture0, uv + offset).rgba;
    float outAlpha = alphaCenter * insideMask;
    eyeView.rgb *= insideMask;
    eyeView.a = outAlpha;

    gl_FragColor = eyeView;
}
